/*****************************************************************************
 * request_handler.c
 * Functions to handle a request from client
 *****************************************************************************
 * Copyright (C) 1998, 1999, 2000, 2001 VideoLAN
 * $Id: request_handler.c,v 1.23 2001/04/29 03:41:49 nitrox Exp $
 *
 * Authors: Laurent Rossier <gunther@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <arpa/inet.h>                                            /* types.h */
#include <sys/socket.h>                                    /* socket, sendto */
#include <stdlib.h>                                                  /* free */
#include <string.h>                                                 /* bzero */
#include <unistd.h>                                                 /* close */
#include <pthread.h>                                                 /* db.h */
#include <semaphore.h>                                               /* db.h */
#include <netdb.h>                                              /* if_snmp.h */
#include <ucd-snmp/asn1.h>                                      /* if_snmp.h */
#include <ucd-snmp/snmp.h>                                      /* if_snmp.h */
#include <ucd-snmp/snmp_impl.h>                                 /* if_snmp.h */
#include <ucd-snmp/default_store.h>                             /* if_snmp.h */
#include <ucd-snmp/snmp_api.h>                                  /* if_snmp.h */
#include <ucd-snmp/snmp_client.h>                               /* if_snmp.h */
#include <ucd-snmp/mib.h>                                       /* if_snmp.h */

#include "../types.h"
#include "../logger.h"

#include "../snmp/snmp_switch.h"                       /* struct SNMP_switch */
#include "../snmp/if_snmp.h"                                /* SNMP_set_vlan */
#include "../db/db.h"                                    /* DB_machines_lock */
#include "../config/config.h"                                     /* vs->cfg */
#include "../vlanserver.h"                                             /* vs */
#include "connect.h"                                     /* CNX_vlb_set_vlan */
#include "server.h"                                 /* struct VS_info_client */

/*****************************************************************************
 * VS_request_handler
 *****************************************************************************
 * Scan the message from the client
 * Send a message to the client
 * Lock the machine and the port in the db
 * Call the SNMP_set_vlan and the CNX_vlb_set_vlan functions
 * Unlock the machine and the port in the db
 *****************************************************************************/
ERR_CODE VS_request_handler(struct VS_info_client * info_client)
{
  VS_MachineId macaddr_db;
  VS_CHANNEL channel_dest;
  VS_VLAN vlan_dest;
  unsigned int rc;
  unsigned int i1, i2, i3, i4, i5;
  unsigned int version_cl;
  unsigned int date_cl;
  int socket_th;
  struct sockaddr_in sa_client;
  struct DB_machines_elt * machines_elt;
  struct SNMP_switch * zwitch;
  struct DB_port * port;
  char macaddr_vlb[18];
  char message[6];

  macaddr_db=0;
  macaddr_vlb[17] = 0;

  /*
   * Analysis of the paquet
   */
  VS_log(LOGDEBUG,SERVER,"%s", info_client->mess);
  if(sscanf(info_client->mess, "%hhu %u %u %s", &channel_dest, &version_cl,\
                                                     &date_cl, macaddr_vlb)!=4)
  {
    VS_log(LOGERROR,SERVER,"Wrong packet received");
    return VS_R_PARSE;
  }
  VS_log(LOGINFO,SERVER,"IP %s mac %s chandest %hhu version %u sessid %u date"\
                                    "cl %u", info_client->ipaddr, macaddr_vlb,\
                   channel_dest, version_cl, info_client->session_id, date_cl);
  
  if(sscanf(macaddr_vlb, "%x:%x:%x:%x:%x:%x", (unsigned *)&macaddr_db,\
                                                   &i1, &i2, &i3, &i4, &i5)!=6)
  {
    VS_log(LOGERROR,SERVER,"Wrong packet received");
    return VS_R_PARSE;
  }
  macaddr_db<<=8;
  macaddr_db|=i1;
  macaddr_db<<=8;
  macaddr_db|=i2;
  macaddr_db<<=8;
  macaddr_db|=i3;
  macaddr_db<<=8;
  macaddr_db|=i4;
  macaddr_db<<=8;
  macaddr_db|=i5;

  
  /*
   * Opening the socket with the client
   */
  socket_th=socket(AF_INET, SOCK_DGRAM, 0);
  if(socket_th==-1)
  {
    VS_log(LOGERROR,SERVER,"Unable to open a socket");
    return VS_R_SOCKET;
  }
  bzero(&sa_client, sizeof(struct sockaddr_in));
  sa_client.sin_family=AF_INET;
  sa_client.sin_port=htons(4312);
  inet_aton(info_client->ipaddr, &(sa_client.sin_addr));

  
  /*
   * Effective part of the request
   */
  
  /* looking for the DB entry of the mac */
  rc=DB_machines_lock(vs->db, macaddr_db, &machines_elt);
  if(!rc)
  {
    VS_log(LOGINFO, SERVER, "switch:%s port:%u",\
                                            VS_SID2A(machines_elt->switch_ip),\
                                                 (unsigned)machines_elt->port);
    
    /* looking of the DB entry of the switch */
    rc=DB_switchs_port_lock(vs->db, machines_elt->switch_ip,\
                                           machines_elt->port, &port, &zwitch);
    if(!rc)
    {
      if(zwitch->ports[machines_elt->port].protection<VS_PL_VIDEOLAN)
      {
        vlan_dest=ChannelToVlan(*(vs->cfg), channel_dest);
        
        /* Asking the SNMP to change VLAN */
        rc=SNMP_set_vlan(vs->snmp, zwitch, machines_elt->port, vlan_dest);
        if(!rc)
        {
          /* Asking the VLB to bridge */
          /* TODO We have here to bridge all macs known on the port */
          CNX_vlb_set_vlan(3, macaddr_vlb, info_client->ipaddr,\
                            vlan_dest, zwitch->ports[machines_elt->port].vlan);
          zwitch->ports[machines_elt->port].vlan=vlan_dest;
          
          /* Changing the DB entry */
          machines_elt->channel=channel_dest;
          VS_log(LOGINFO, SERVER, "suceed to change client's channel");
        }
        else
        {
          /* Failed because of SNMP */
          VS_log(LOGERROR,SERVER,"Unable to call the SNMP_set_vlan function");
          VS_log(LOGERROR,SERVER,"Request interrupted");
          sprintf(message, "snmp");
          sendto(socket_th, &message, sizeof(message), 0,\
                     (struct sockaddr *)(&sa_client), sizeof(struct sockaddr));
        }
        DB_switchs_port_unlock(vs->db, port);
      }
      else
      {
        /* Failed because of permission */
        VS_log(LOGWARNING,SERVER,"Permission denied to change the VLAN of a "\
                                                             "protected port");
        sprintf(message, "port");
        sendto(socket_th, &message, sizeof(message), 0,\
                     (struct sockaddr *)(&sa_client), sizeof(struct sockaddr));
      }
    }
    else
    {
      /* Failed because of db */
      VS_log(LOGINFO,SERVER,"Pb with the DB_switchs_port_lock function");
      sprintf(message, "db");
      sendto(socket_th, &message, sizeof(message), 0,\
                     (struct sockaddr *)(&sa_client), sizeof(struct sockaddr));
    }

    DB_machines_unlock(vs->db, machines_elt);
  }
  else
  {
    /* Failed because of mac entry in db not found */
    VS_log(LOGERROR,SERVER,"MAC not found in database");
    sprintf(message, "mac");
    sendto(socket_th, &message, sizeof(message), 0,\
                     (struct sockaddr *)(&sa_client), sizeof(struct sockaddr));
    close(socket_th);
    free(info_client);
    return rc;
  }
  close(socket_th);
  free(info_client);
  return 0;
}
