///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

/*Prog:lehmer
NAME: @code{lehmer} - make the A(i,j) = i/j for j>=i N-by-N matrix 
SYNOPSIS:
  @example
    lehmer @var{N} [-check]
  @end example

DESCRIPTION:       
    
    Demonstration of "dns", "csr" and input output usage.

    a = lehmer(@var{N}) is the symmetric positive definite @var{N}-by-@var{N} matrix with
    A(i,j) = i/j for j>=i.

    A is totally nonnegative.  inv(A) is tridiagonal, and explicit
    formulas are known for its entries.

    @var{N} <= cond(A) <= 4*@var{N}*@var{N}.

    References:
    @quotation
    M. Newman and J. Todd, The evaluation of matrix inversion
    programs, J. Soc. Indust. Appl. Math., 6 (1958), pp. 466-476.
    
    Solutions to problem E710 (proposed by D.H. Lehmer): The inverse
    of a matrix, Amer. Math. Monthly, 53 (1946), pp. 534-535.
    @end quotation

EXAMPLE:
  Enter command as:
  @example
    	lehmer 20 | hb2ps | ghostview -
  @end example
  @noindent
  and check the properties by:
  @example
    	lehmer 8 -check | octave
  @end example

SEE ALSO:
    "hb2ps"(1), class "csr", class "dns", class "ioskit"

AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
DATE: 
    20 february 1997

METHOD: @lehmer
End:
*/
//<lehmer:
# include "rheolef/skit.h"
using namespace rheolef;
using namespace std;

csr<Float> lehmer (unsigned int N)
{
  dns<Float> a (N,N);
  for (unsigned int i = 0; i < N; i++) 
    for (unsigned int j = 0; j < N; j++)
      a(i,j) = j>=i ? Float(i+1)/Float(j+1) : Float(j+1)/Float(i+1);
  return csr<Float>(a);
}

int main(int argc, char *argv[])
{
  if (argc < 2 || atoi(argv[1]) < 1) {

    cerr << argv[0] << ": usage: " 
         << argv[0] << " N [-check]\n";
    
  } else if (argc < 3) { // print Harwell-Boeing output
        
    cout << lehmer (atoi(argv[1]));

  } else {   // print matlab code
        
    cout << ml 
         << "A = " << lehmer (atoi(argv[1])) << endl
         << "inv_A=inv(A)"   << endl
         << "N=" << argv[1]  << endl
         << "cond_A=cond(A)" << endl
         << "N_N_4=4*N*N"    << endl;
  }
  return 0;
}
//>lehmer:
