#ifndef _RHEO_DIV_S_H
#define _RHEO_DIV_S_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:div_s
NAME: @code{div_s} -- divergence operator on a surface
@bfindex div_s
@bfindex grad_s
@apindex P1
@cindex level set method
SYNOPSIS:
    @example
        form(const space& Bvh, const space& Bh, "div_s", phi_h);
    @end example
DESCRIPTION:       
    @noindent
    On a surface Gamma defined as the zero of a level-set function,
    assembly the matrix associated to the following form:
@iftex
@tex
    $$
   	b({\bf v},q) = \int_\Gamma {\rm div}_s({\bf v}) \, q\, ds
    $$
@end tex
@end iftex
@ifnottex
    @example
                 /
                 |
   	b(v,q) = | div_s(v) q ds
                 |
                 / Gamma
    @end example
@end ifnottex
    
    @noindent
    The Bh space may be a @code{P1} finite element space on the band arround the zero level set.
    The Bvh space may be the "vector" variant of Bh.
    Notice that elements different from P1 are not yet supported.
REMARK:
    The div_s and the grad_s forms are not adjoint and thus the corresponding
    discrete operators can not be obtained by transposition.
    Recall the Green formula:
@ifnottex
    @example
         /                 /                  /
        |                  |                  |
   	| div_s(v) q ds +  | v.grad_s(q) ds = | div_s(n) v.n q ds
        |                  |                  |
        / Gamma           / Gamma             / Gamma
    @end example
@end ifnottex
@iftex
@tex
    $$
   	\int_\Gamma {\rm div}_s({\bf v}) \, q\, ds
   	+
        \int_\Gamma {\bf v} . \nabla_s q\, ds
	=
        \int_\Gamma ({\rm div_s}\,{\bf n}) \, {\bf v}.{\bf n} \, q \, ds
    $$
    for all ${\bf v}$ and $q$, where ${\bf n}$ is an unit normal on $\Gamma$.
@end tex
@end iftex
    Notice that the right-hand side is non-zero when the curvature div_s(n) is non-zero.

EXAMPLE:
    @example
         space Vh (lambda, "P1");
         field phi_h = interpolate(Vh, phi);
         geo beta = banded_level_set (phi_h);
         space Bh (beta, "P1");
         Bh.block("isolated");
         space Bh (beta, "P1", "vector");
         field phi_bh = interpolate(Bh, phi);
         form b  (Bvh, Bh, "div_s", phi_bh);
    @end example
AUTHOR: Mahamar Dicko, Jocelyn Etienne, Pierre Saramito
DATE:
    18 april 2011
End:
*/

#include "rheolef/form_element_rep.h"
namespace rheolef { 

class div_s: public form_element_rep {
public:
// allocator:

    div_s() : form_element_rep() {}

// virtual accessor:

    void operator() (const geo_element& K, ublas::matrix<Float>& m) const;
    void check_after_initialize () const;
};
}// namespace rheolef
#endif // _RHEO_DIV_S_H
