#!vtk
#
# This file is part of Rheolef.
#
# Copyright (C) 2000-2009 Pierre Saramito 
#
# Rheolef is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Rheolef is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Rheolef; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------
#
# volume render for scalar data on unstructured 3d meshes
#
# date: 30 octobre 2000
#
# author: Pierre.Saramito@imag.fr
#
# NOTE: tested on vtk-3.1.2
#
# ----------------------------------------------------------
# usage
# ----------------------------------------------------------
#
# for inlusion in a 'wrapper.tcl' file:
#
#	set file_name "my-file.vtk"
#	set color_wheel_file "vtk_color_wheel.ppm"
#	source "vtk_interactor.tcl"
#	source "vtk_widget_object.tcl"
#	source "vtk_volume_interactor.tcl"
#	source "vtk_volume.tcl"
#
# then
#	vtk wrapper.tcl
# 
# ----------------------------------------------------------
# internal parameters
# ----------------------------------------------------------

set n_image_resolution 20;  # for numerisation
set n_resolution       100; # for scalar selection

set axes_color      "1. 1. 1.";

set envelop_opacity 0.2;
set envelop_color   "1. 1. 1.";

# for MIP and compisite
set digitalize_max_distance 0.27;

# ----------------------------------------------------------
# read the file
# ----------------------------------------------------------

vtkUnstructuredGridReader reader;
    reader SetFileName $file_name
    reader Update;

set bounds [[reader GetOutput] GetBounds]
scan $bounds "%f %f %f %f %f %f" minx maxx miny maxy minz maxz
set frange [[[[reader GetOutput] GetPointData] GetScalars] GetRange]
set fmin [lindex $frange 0]
set fmax [lindex $frange 1]

set xmid [expr ($maxx + $minx)/2.0];
set ymid [expr ($maxy + $miny)/2.0];
set zmid [expr ($maxz + $minz)/2.0];
set xdiff [expr $maxx - $minx];
set ydiff [expr $maxy - $miny];
set zdiff [expr $maxz - $minz];

# ----------------------------------------------------------

proc change_active_controls { v1 v2 v3 } {
    # ignore the arguments... but don't remove them (needed for trace callback)
    
    global control_menu_value

    if [ winfo ismapped .top.f1.f1.f1.mip ] {
	pack forget .top.f1.f1.f1.mip 
    }

    if [ winfo ismapped .top.f1.f1.f1.comp ] {
	pack forget .top.f1.f1.f1.comp 
    }

    if [ winfo ismapped .top.f1.f1.f1.iso ] {
	pack forget .top.f1.f1.f1.iso 
    }

    if [ winfo ismapped .top.f1.f1.f1.slice ] {
	pack forget .top.f1.f1.f1.slice 
    }

    if { $control_menu_value == "MIP Controls" } {
	pack .top.f1.f1.f1.mip -side top -expand 1 -fill both -pady 20
    } elseif  { $control_menu_value == "Composite Controls" } {
	pack .top.f1.f1.f1.comp -side top -expand 1 -fill both -pady 20
    } elseif  { $control_menu_value == "Isosurface Controls" } {
	pack .top.f1.f1.f1.iso -side top -expand 1 -fill both -pady 20
    } elseif  { $control_menu_value == "Slice Viewer Controls" } {
	pack .top.f1.f1.f1.slice -side top -expand 1 -fill both -pady 20
    }
}
proc set_isosurface_value { } {
  global iso_value
  global iso_filter
  global ren_win4

  iso_filter SetValue 0 $iso_value
  $ren_win4 Render  
}
# --------------------------------------------------------
# set x y and z slice value ; refresh the slice view
# --------------------------------------------------------

proc set_z_slice_value { } {
  global z_slice_value
  global xmid ymid
  global ren_win1

  plane1 SetOrigin $xmid $ymid $z_slice_value;
  if { [[ren1 GetActors] GetNumberOfItems] > 0 } {
      $ren_win1 Render
  }
}
proc set_y_slice_value { } {
  global y_slice_value
  global xmid zmid
  global ren_win1

  plane2 SetOrigin $xmid $y_slice_value $zmid;
  if { [[ren1 GetActors] GetNumberOfItems] > 0 } {
      $ren_win1 Render
  }
}
proc set_x_slice_value { } {
  global x_slice_value
  global ymid zmid
  global ren_win1

  plane3 SetOrigin $x_slice_value $ymid $zmid;
  if { [[ren1 GetActors] GetNumberOfItems] > 0 } {
      $ren_win1 Render
  }
}
# ---------------------------------------------
# Create the top level stuff and basic frames
# ---------------------------------------------

wm withdraw .

toplevel .top -visual best

frame .top.f1
frame .top.f2 -bg #999999

frame .top.f1.f1 -bd 4 -bg #770099 -relief ridge
frame .top.f1.f2

pack .top.f1 .top.f2 -side top -expand 1 -fill both

pack .top.f1.f1 .top.f1.f2 -side left -expand 1 -fill both



### Create the render windows

frame .top.f1.f2.f1 -bd 0
frame .top.f1.f2.f2 -bd 0

frame .top.f1.f2.f1.f1 -bd 4 -bg #770099 -relief ridge 
frame .top.f1.f2.f1.f2 -bd 4 -bg #770099 -relief ridge 

frame .top.f1.f2.f2.f1 -bd 4 -bg #770099 -relief ridge 
frame .top.f1.f2.f2.f2 -bd 4 -bg #770099 -relief ridge 

label .top.f1.f2.f1.f1.label -text "Slice Viewer" \
	-bg #999999 -fg #440066 \
	-font {Helvetica -12 bold}

label .top.f1.f2.f1.f2.label -text "MIP Rendering" \
	-bg #999999 -fg #440066 \
	-font {Helvetica -12 bold}

label .top.f1.f2.f2.f1.label -text "Composite Rendering" \
	-bg #999999 -fg #440066 \
	-font {Helvetica -12 bold}

label .top.f1.f2.f2.f2.label -text "Isosurface Rendering" \
	-bg #999999 -fg #440066 \
	-font {Helvetica -12 bold}

vtkTkRenderWidget .top.f1.f2.f1.f1.ren -width 256 -height 256 
    BindTkRenderWidget .top.f1.f2.f1.f1.ren

vtkTkRenderWidget .top.f1.f2.f1.f2.ren -width 256 -height 256 
    BindTkRenderWidget .top.f1.f2.f1.f2.ren

vtkRenderWindow ren_win
vtkTkRenderWidget .top.f1.f2.f2.f1.ren -rw ren_win -width 256 -height 256 
    BindTkRenderWidget .top.f1.f2.f2.f1.ren

vtkTkRenderWidget .top.f1.f2.f2.f2.ren -width 256 -height 256 
    BindTkRenderWidget .top.f1.f2.f2.f2.ren

pack .top.f1.f2.f1 .top.f1.f2.f2 -side top

pack .top.f1.f2.f1.f1 .top.f1.f2.f1.f2 -side left
pack .top.f1.f2.f2.f1 .top.f1.f2.f2.f2 -side left

pack .top.f1.f2.f1.f1.label .top.f1.f2.f1.f1.ren -side top -expand 1 -fill both
pack .top.f1.f2.f1.f2.label .top.f1.f2.f1.f2.ren -side top -expand 1 -fill both
pack .top.f1.f2.f2.f1.label .top.f1.f2.f2.f1.ren -side top -expand 1 -fill both
pack .top.f1.f2.f2.f2.label .top.f1.f2.f2.f2.ren -side top -expand 1 -fill both

set ren_win1 [.top.f1.f2.f1.f1.ren GetRenderWindow]
vtkRenderer ren1
$ren_win1 AddRenderer ren1

set ren_win2 [.top.f1.f2.f1.f2.ren GetRenderWindow]
vtkRenderer ren2
$ren_win2 AddRenderer ren2

set ren_win3 [.top.f1.f2.f2.f1.ren GetRenderWindow]
vtkRenderer ren3
$ren_win3 AddRenderer ren3

set ren_win4 [.top.f1.f2.f2.f2.ren GetRenderWindow]
vtkRenderer ren4
$ren_win4 AddRenderer ren4

$ren_win1 SetSize 256 256
$ren_win2 SetSize 256 256
$ren_win3 SetSize 256 256
$ren_win4 SetSize 256 256


### Create the help message area

label .top.f2.help_label -text "Rheolef volume rendering" \
	-bg #999999 -fg #440066 \
	-font {Helvetica -12 bold}

label .top.f2.help_message \
	-bg #999999 -fg #000000 \
	-font {Helvetica -12 bold} \
	-text \
"Left Mouse Button: rotate
Shift-Left or Middle Mouse Button: pan
Right Mouse Button: zoom
r key: reset view"

button .top.f2.exit -text Exit -fg #999999 -bg #440066 \
	-activeforeground #440066 -activebackground #999999 \
	-highlightthickness 0 -bd 4 -command { exit } \
	-font {Helvetica -12 bold}

pack .top.f2.help_label .top.f2.help_message .top.f2.exit -padx 5 -pady 5 \
	-expand 1 -fill both


### Create the control options area

frame .top.f1.f1.f1 -bg #999999

label .top.f1.f1.f1.label -text "Control Options" \
	-bg #999999 -fg #000000 \
	-font {Helvetica -12 bold}

set control_menu_value {Slice Viewer Controls}

set control_menu [tk_optionMenu .top.f1.f1.f1.menu control_menu_value \
        {Slice Viewer Controls} \
	{MIP Controls} {Composite Controls} {Isosurface Controls} ]

.top.f1.f1.f1.menu configure -fg #999999 -bg #440066 \
	-activeforeground #440066 -activebackground #999999 \
	-highlightthickness 0 -bd 4 \
	-font {Helvetica -12 bold} 


$control_menu configure -bg #999999 -fg #440066 \
	-activeforeground #220033 -activebackground #bbbbbb \
	-font {Helvetica -12 bold} 

# trace the variable that is set by the option menu
trace variable control_menu_value w change_active_controls

pack .top.f1.f1.f1 -side top -expand 1 -fill both 
pack .top.f1.f1.f1.label  -side top -expand 0 -fill both -padx 50 -pady 5
pack .top.f1.f1.f1.menu -side top -expand 0 -fill none -padx 2 -pady 5

# ------------------------------------------------
# Create the control options area for MIP
# ------------------------------------------------

set ww .top.f1.f1.f1

frame $ww.mip -bg #999999

label $ww.mip.l1 -text "Interpolation Type:" -bg #999999 -fg #000000 \
	-font {Helvetica -12 bold}

set mip_interpolation_type 0

radiobutton $ww.mip.rb1 -text "Nearest Neighbor" \
	-variable mip_interpolation_type -value 0 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ mip_prop SetInterpolationTypeToNearest; $ren_win2 Render }

radiobutton $ww.mip.rb2 -text "Trilinear" \
	-variable mip_interpolation_type -value 1 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ mip_prop SetInterpolationTypeToLinear; $ren_win2 Render }
 
pack $ww.mip.l1 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w

pack $ww.mip.rb1 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w
pack $ww.mip.rb2 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w

label $ww.mip.l2 -text "Value To Color Mapping:" -bg #999999 -fg #000000 \
	-font {Helvetica -12 bold}

set mip_color_type 0

radiobutton $ww.mip.rb3 -text "Greyscale" \
	-variable mip_color_type -value 0 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ mip_prop SetColor gtfun; $ren_win2 Render }

radiobutton $ww.mip.rb4 -text "Color" \
	-variable mip_color_type -value 1 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ mip_prop SetColor ctfun; $ren_win2 Render }

pack $ww.mip.l2 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w

pack $ww.mip.rb3 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w
pack $ww.mip.rb4 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w


label $ww.mip.l3 -text "Intensity:" -bg #999999 -fg #000000 \
 	-font {Helvetica -12 bold}
set f_step [expr 1.0/$n_resolution ]
scale $ww.mip.s3 -length 120 -from 0 -to 1 -resolution $f_step \
	-bg #999999 -fg #770099 -variable mip_intensity \
	-font {Helvetica -12 bold} \
	-orient horizontal -highlightthickness 0
bind $ww.mip.s3 <ButtonRelease> { set_mip_intensity }
pack $ww.mip.l3 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w
pack $ww.mip.s3 -side top -padx 22 -pady 10 -expand 0 -fill none -anchor n

label $ww.mip.l4 -text "Opacity:" -bg #999999 -fg #000000 \
 	-font {Helvetica -12 bold}
set f_step [expr 1.0/$n_resolution ]
scale $ww.mip.s4 -length 120 -from 0 -to 1 -resolution $f_step \
	-bg #999999 -fg #770099 -variable mip_opacity \
	-font {Helvetica -12 bold} \
	-orient horizontal -highlightthickness 0
bind $ww.mip.s4 <ButtonRelease> { set_mip_opacity }
pack $ww.mip.l4 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w
pack $ww.mip.s4 -side top -padx 22 -pady 10 -expand 0 -fill none -anchor n

button $ww.mip.save_as -text "Save image as \"mip.${default_image_file_format}\"" \
    -fg #999999 -bg #440066 -activeforeground #440066 -activebackground #999999 \
    -highlightthickness 0 -bd 4 -command {save_as $ren_win2 "mip"}
pack $ww.mip.save_as -side top -expand 1 -fill x

# ------------------------------------------------
# Create the control options area for Composite
# ------------------------------------------------

set ww .top.f1.f1.f1

frame $ww.comp -bg #999999

label $ww.comp.l1 -text "Interpolation Type:" -bg #999999 -fg #000000 \
	-font {Helvetica -12 bold}

set composite_interpolation_type 0

radiobutton $ww.comp.rb1 -text "Nearest Neighbor" \
	-variable composite_interpolation_type -value 0 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ composite_prop SetInterpolationTypeToNearest ; $ren_win3 Render }


radiobutton $ww.comp.rb2 -text "Trilinear" \
	-variable composite_interpolation_type -value 1 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ composite_prop SetInterpolationTypeToLinear ; $ren_win3 Render }
 
pack $ww.comp.l1 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w

pack $ww.comp.rb1 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w
pack $ww.comp.rb2 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w

label $ww.comp.l2 -text "Value To Color Mapping:" -bg #999999 -fg #000000 \
	-font {Helvetica -12 bold} \

set composite_color_type 1

radiobutton $ww.comp.rb3 -text "Greyscale" \
	-variable composite_color_type -value 0 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ composite_prop SetColor gtfun; $ren_win3 Render }


radiobutton $ww.comp.rb4 -text "Color" \
	-variable composite_color_type -value 1 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ composite_prop SetColor ctfun; $ren_win3 Render }

 
pack $ww.comp.l2 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w

pack $ww.comp.rb3 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w
pack $ww.comp.rb4 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w

label $ww.comp.l3 -text "Shading:" -bg #999999 -fg #000000 \
	-font {Helvetica -12 bold}

set composite_shade_type 0

radiobutton $ww.comp.rb5 -text "Off" \
	-variable composite_shade_type -value 0 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ composite_prop ShadeOff ; tfun AddPoint 255.0 0.2 ; $ren_win3 Render }


radiobutton $ww.comp.rb6 -text "On" \
	-variable composite_shade_type -value 1 \
	-bg #999999 -fg #000000 -selectcolor #aa00ff \
	-highlightthickness 0 -activebackground #999999 \
	-font {Helvetica -12 bold} \
	-activeforeground #7700ff -command \
	{ composite_prop ShadeOn ; tfun AddPoint 255.0 0.4 ; $ren_win3 Render }
 
 
pack $ww.comp.l3 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w

pack $ww.comp.rb5 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w
pack $ww.comp.rb6 -side top -padx 22 -pady 2 -expand 0 -fill none -anchor w

label $ww.comp.l7 -text "Opacity:" -bg #999999 -fg #000000 \
 	-font {Helvetica -12 bold}
set f_step [expr 1.0/$n_resolution ]
scale $ww.comp.s7 -length 120 -from 0 -to 1 -resolution $f_step \
	-bg #999999 -fg #770099 -variable composite_opacity \
	-font {Helvetica -12 bold} \
	-orient horizontal -highlightthickness 0
bind $ww.comp.s7 <ButtonRelease> { set_composite_opacity }
pack $ww.comp.l7 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w
pack $ww.comp.s7 -side top -padx 22 -pady 10 -expand 0 -fill none -anchor n

button $ww.comp.save_as -text "Save image as \"comp.${default_image_file_format}\"" \
    -fg #999999 -bg #440066 -activeforeground #440066 -activebackground #999999 \
    -highlightthickness 0 -bd 4 -command {save_as $ren_win3 "comp"}
pack $ww.comp.save_as -side top -expand 1 -fill x

# ------------------------------------------------
# Create the control options area for Isosurface
# ------------------------------------------------

set ww .top.f1.f1.f1

frame $ww.iso -bg #999999

label $ww.iso.l3 -text "Isovalue:" -bg #999999 -fg #000000 \
 	-font {Helvetica -12 bold}
set iso_value [expr ($fmin + $fmax)/2. ]
set f_step [expr ($fmax - $fmin)/$n_resolution ]
scale $ww.iso.s1 -length 120 -from $fmin -to $fmax -resolution $f_step \
	-bg #999999 -fg #770099 -variable iso_value \
	-font {Helvetica -12 bold} \
	-orient horizontal -highlightthickness 0
bind $ww.iso.s1 <ButtonRelease> { set_isosurface_value }
pack $ww.iso.l3 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w
pack $ww.iso.s1 -side top -padx 22 -pady 10 -expand 0 -fill none -anchor n

label $ww.iso.l4 -text "Surface Color:" -bg #999999 -fg #000000 \
	-font {Helvetica -12 bold}
image create photo color_wheel -file $color_wheel_file
label $ww.iso.l5 -image color_wheel -highlightthickness 0
pack $ww.iso.l4 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor w
pack $ww.iso.l5 -side top -padx 2 -pady 2 -expand 0 -fill none -anchor n
bind $ww.iso.l5 <Button-1> {
    scan [color_wheel get %x %y] "%%f %%f %%f" r g b
    set r [expr $r / 255.0]
    set g [expr $g / 255.0]
    set b [expr $b / 255.0]
    set iso_color "$r $g $b";
    eval [iso_actor GetProperty] SetColor $iso_color
    $ren_win4 Render
}
button $ww.iso.save_as -text "Save image as \"iso.${default_image_file_format}\"" \
    -fg #999999 -bg #440066 -activeforeground #440066 -activebackground #999999 \
    -highlightthickness 0 -bd 4 -command {save_as $ren_win4 "iso"}
pack $ww.iso.save_as -side top -expand 1 -fill x
# ----------------------------------------------------------
# Create the control options area for the slice viewer
# ----------------------------------------------------------

set ww .top.f1.f1.f1

frame $ww.slice -bg #999999

frame $ww.slice.x -bg #999999
frame $ww.slice.y -bg #999999
frame $ww.slice.z -bg #999999

pack $ww.slice.x $ww.slice.y $ww.slice.z -side top -padx 2 -pady 5 \
	-expand 0 -fill none

set x_slice_value [expr ($maxx + $minx) / 2.0]
set x_step [expr ($maxx - $minx)/$n_resolution]
label $ww.slice.x.l1 -text "X Slice:" -bg #999999 -fg #000000
scale $ww.slice.x.s1 -length 100 -from $minx -to $maxx -resolution $x_step \
	-bg #999999 -fg #770099 -variable x_slice_value \
	-orient horizontal -highlightthickness 0
pack $ww.slice.x.l1 -side left -padx 5 -pady 2 -expand 0 -fill none -anchor w
pack $ww.slice.x.s1 -side left -padx 5 -pady 2 -expand 0 -fill none -anchor w
bind $ww.slice.x.s1 <ButtonRelease> { set_x_slice_value }


set y_slice_value [expr $ydiff / 2.0]
set y_step [expr ($maxy - $miny)/$n_resolution]
label $ww.slice.y.l1 -text "Y Slice:" -bg #999999 -fg #000000
scale $ww.slice.y.s1 -length 100 -from $miny -to $maxy -resolution $y_step \
	-bg #999999 -fg #770099 -variable y_slice_value \
	-orient horizontal -highlightthickness 0
pack $ww.slice.y.l1 -side left -padx 5 -pady 2 -expand 0 -fill none -anchor w
pack $ww.slice.y.s1 -side left -padx 5 -pady 2 -expand 0 -fill none -anchor w
bind $ww.slice.y.s1 <ButtonRelease> { set_y_slice_value }


set z_slice_value [expr $zdiff / 2.0]
set z_step [expr ($maxz - $minz)/$n_resolution]
label $ww.slice.z.l1 -text "Z Slice:" -bg #999999 -fg #000000
scale $ww.slice.z.s1 -length 100 -from $minz -to $maxz -resolution $z_step \
	-bg #999999 -fg #770099 -variable z_slice_value \
	-orient horizontal -highlightthickness 0
pack $ww.slice.z.l1 -side left -padx 5 -pady 2 -expand 0 -fill none -anchor w
pack $ww.slice.z.s1 -side left -padx 5 -pady 2 -expand 0 -fill none -anchor w
bind $ww.slice.z.s1 <ButtonRelease> { set_z_slice_value }
pack .top.f1.f1.f1.slice -side top -expand 1 -fill both -pady 20

button $ww.slice.save_as -text "Save image as \"slice.${default_image_file_format}\"" \
    -fg #999999 -bg #440066 -activeforeground #440066 -activebackground #999999 \
    -highlightthickness 0 -bd 4 -command {save_as $ren_win1 "slice"}
pack $ww.slice.save_as -side top -expand 1 -fill x

# -----------------------------------------------------
# outline
# -----------------------------------------------------

vtkCubeSource outline

    outline SetXLength $xdiff;
    outline SetYLength $ydiff;
    outline SetZLength $zdiff;
    outline SetCenter  [ expr $xmid ] \
                       [ expr $ymid ] \
                       [ expr $zmid ]

vtkPolyDataMapper outline_mapper
    outline_mapper SetInput [outline GetOutput]
    outline_mapper ImmediateModeRenderingOn

vtkActor outline_actor
    outline_actor SetMapper outline_mapper
    [outline_actor GetProperty] SetColor .7 0 .9
    [outline_actor GetProperty] SetAmbient  1
    [outline_actor GetProperty] SetDiffuse  0
    [outline_actor GetProperty] SetSpecular 0
    [outline_actor GetProperty] SetRepresentationToWireframe

# -----------------------------------------------------
# envelop
# -----------------------------------------------------

vtkDataSetMapper envelop_mapper;
    envelop_mapper SetInput [reader GetOutput];
    envelop_mapper ScalarVisibilityOff;
vtkActor envelop_actor;
    envelop_actor SetMapper envelop_mapper;
    eval [envelop_actor GetProperty] SetColor $envelop_color;
    [envelop_actor GetProperty] SetRepresentationToWireframe;
    [envelop_actor GetProperty] SetOpacity $envelop_opacity;
    [envelop_actor GetProperty] SetRepresentationToSurface;
;

# ---------------------------------------------------
# isosurfaces
# ---------------------------------------------------

vtkContourFilter iso_filter
    iso_filter SetInput [reader GetOutput]
    iso_filter SetValue 0 $iso_value
    iso_filter ComputeGradientsOff;
    iso_filter ComputeNormalsOff;
    iso_filter ComputeScalarsOff;

vtkPolyDataNormals normals
    normals SetInput [iso_filter GetOutput]
    normals SetFeatureAngle 45
vtkPolyDataMapper iso_mapper
    iso_mapper SetInput [normals GetOutput]
    iso_mapper ScalarVisibilityOff

set iso_color "1.0000 0.8941 0.7686";

vtkActor iso_actor
    iso_actor SetMapper iso_mapper
    eval [iso_actor GetProperty] SetColor $iso_color

# -----------------------------
# cut: this is the first plane
# -----------------------------

vtkPlane plane1;
    eval plane1 SetOrigin [[reader GetOutput] GetCenter];
    plane1 SetNormal 0 0 1;
vtkCutter plane_cut1;
    plane_cut1 SetInput [reader GetOutput];
    plane_cut1 SetCutFunction plane1;
vtkDataSetMapper map_fill1;
    map_fill1 SetInput [plane_cut1 GetOutput];
    eval map_fill1 SetScalarRange [[reader GetOutput] GetScalarRange]

vtkActor plane_cut_actor1;
    plane_cut_actor1 SetMapper map_fill1;
    [plane_cut_actor1 GetProperty] SetOpacity 1.0

vtkOutlineFilter outline_filter1
    outline_filter1 SetInput [plane_cut1 GetOutput]
vtkPolyDataMapper map_outline1
    map_outline1 SetInput [outline_filter1 GetOutput]
vtkActor plane_outline_actor1
    plane_outline_actor1 SetMapper map_outline1
    [plane_outline_actor1 GetProperty] SetColor 0 0.5 1.0

# -----------------------------
# cut: this is the second plane
# -----------------------------

vtkPlane plane2;
    eval plane2 SetOrigin [[reader GetOutput] GetCenter];
    plane2 SetNormal 0 1 0;
vtkCutter plane_cut2;
    plane_cut2 SetInput [reader GetOutput];
    plane_cut2 SetCutFunction plane2;
vtkDataSetMapper map_fill2;
    map_fill2 SetInput [plane_cut2 GetOutput];
    eval map_fill2 SetScalarRange [[reader GetOutput] GetScalarRange]
vtkActor plane_cut_actor2;
    plane_cut_actor2 SetMapper map_fill2;
    [plane_cut_actor2 GetProperty] SetOpacity 1.0

vtkOutlineFilter outline_filter2
    outline_filter2 SetInput [plane_cut2 GetOutput]
vtkPolyDataMapper map_outline2
    map_outline2 SetInput [outline_filter2 GetOutput]
vtkActor plane_outline_actor2
    plane_outline_actor2 SetMapper map_outline2
    [plane_outline_actor2 GetProperty] SetColor 0 0.5 1.0

# -----------------------------
# cut: this is the third plane
# -----------------------------

vtkPlane plane3;
    eval plane3 SetOrigin [[reader GetOutput] GetCenter];
    plane3 SetNormal 1 0 0;
vtkCutter plane_cut3;
    plane_cut3 SetInput [reader GetOutput];
    plane_cut3 SetCutFunction plane3;
vtkDataSetMapper map_fill3;
    map_fill3 SetInput [plane_cut3 GetOutput];
    eval map_fill3 SetScalarRange [[reader GetOutput] GetScalarRange]
vtkActor plane_cut_actor3;
    plane_cut_actor3 SetMapper map_fill3;
    [plane_cut_actor3 GetProperty] SetOpacity 1.0

vtkOutlineFilter outline_filter3
    outline_filter3 SetInput [plane_cut3 GetOutput]
vtkPolyDataMapper map_outline3
    map_outline3 SetInput [outline_filter3 GetOutput]
vtkActor plane_outline_actor3
    plane_outline_actor3 SetMapper map_outline3
    [plane_outline_actor3 GetProperty] SetColor 0 0.5 1.0

# -------------------------------------------------------
# conversion to 3d image with char values: for mip & comp
# -------------------------------------------------------

# convert unstructured data to structured points (float valued)
# and then to image (char valued)

vtkShepardMethod ud2sp
   ud2sp SetInput [reader GetOutput];
   ud2sp SetModelBounds $minx $maxx  $miny $maxy  $minz $maxz
   ud2sp SetSampleDimensions $n_image_resolution $n_image_resolution $n_image_resolution
   ud2sp SetMaximumDistance $digitalize_max_distance
   ud2sp SetNullValue $fmin
   ud2sp Update

vtkImageThreshold low_filter
	low_filter SetInput [ud2sp GetOutput]
	low_filter ThresholdByLower $fmin
	low_filter SetInValue       $fmin

vtkImageThreshold up_filter
	up_filter SetInput [low_filter GetOutput]
	up_filter ThresholdByUpper $fmax
	up_filter SetInValue       $fmax

vtkImageShiftScale image_filter
        image_filter SetInput [up_filter GetOutput]
        image_filter SetShift [expr 0. - $fmin]
        image_filter SetScale [expr 255.0/($fmax-$fmin)]
        image_filter SetOutputScalarTypeToUnsignedChar
        image_filter ClampOverflowOn

# ---------------------------------------------------
# mip
# ---------------------------------------------------

vtkVolumeRayCastMIPFunction  mip_func
        mip_func SetMaximizeMethodToScalarValue

vtkVolumeRayCastMapper mip_volmap
        mip_volmap SetInput [image_filter GetOutput]
        mip_volmap SetVolumeRayCastFunction mip_func

vtkPiecewiseFunction gtfun;
        gtfun AddPoint  0     0.0
        gtfun AddPoint  255   1.0

vtkPiecewiseFunction mip_tfun;
        mip_tfun AddPoint  0     0.0
        mip_tfun AddPoint  255   1.0

vtkVolumeProperty mip_prop
        mip_prop SetColor         gtfun
        mip_prop SetScalarOpacity mip_tfun

vtkVolume mip_volume
        mip_volume SetMapper   mip_volmap
        mip_volume SetProperty mip_prop

set mip_intensity 0.5
proc set_mip_intensity {} {
    	global mip_intensity
        global ren_win2
        set x_middle [expr int(255*(1-$mip_intensity))]
        set y_middle [expr $mip_intensity]
	gtfun RemoveAllPoints
    	gtfun AddPoint  0         0.0
    	gtfun AddPoint  $x_middle $y_middle
        gtfun AddPoint  255       1.0
	# puts "set_mip_intensity $x_middle $y_middle"
        $ren_win2 Render
}
set mip_opacity 0.5
proc set_mip_opacity {} {
    	global mip_opacity
        global ren_win2
        set x_middle [expr int(255*(1-$mip_opacity))]
        set y_middle [expr $mip_opacity]
	mip_tfun RemoveAllPoints
    	mip_tfun AddPoint  0         0.0
    	mip_tfun AddPoint  $x_middle $y_middle
        mip_tfun AddPoint  255       1.0
	# puts "set_mip_opacity $x_middle $y_middle"
        $ren_win2 Render
}
# ---------------------------------------------------
# comp
# ---------------------------------------------------

vtkColorTransferFunction ctfun

if { [get_vtk_major_version]  > 3 || ([get_vtk_major_version] == 3 && [get_vtk_minor_version] >= 2) } {

    	ctfun AddRGBPoint      0.0 	0.0 0.0 0.0
    	ctfun AddRGBPoint     64.0 	1.0 0.0 0.0
    	ctfun AddRGBPoint    128.0 	0.0 0.0 1.0
    	ctfun AddRGBPoint    192.0 	0.0 1.0 0.0
    	ctfun AddRGBPoint    255.0 	0.0 0.2 0.0
} else {
	# older versions
        ctfun AddRedPoint      0.0 0.0
        ctfun AddRedPoint     64.0 1.0
        ctfun AddRedPoint    128.0 0.0
        ctfun AddRedPoint    255.0 0.0
        ctfun AddBluePoint    0.0 0.0
        ctfun AddBluePoint   64.0 0.0
        ctfun AddBluePoint  128.0 1.0
        ctfun AddBluePoint  192.0 0.0
        ctfun AddBluePoint  255.0 0.0
        ctfun AddGreenPoint     0.0 0.0
        ctfun AddGreenPoint   128.0 0.0
        ctfun AddGreenPoint   192.0 1.0
        ctfun AddGreenPoint   255.0 0.2
}
set composite_opacity 0.8;

vtkPiecewiseFunction tfun;
        tfun AddPoint  20   0.0
        tfun AddPoint  255  $composite_opacity

vtkVolumeProperty composite_prop
        composite_prop SetColor ctfun
        composite_prop SetScalarOpacity tfun

vtkVolumeRayCastCompositeFunction  composite_func

vtkVolumeRayCastMapper composite_volmap
        composite_volmap SetInput [image_filter GetOutput]
        composite_volmap SetVolumeRayCastFunction composite_func

vtkVolume composite_volume
        composite_volume SetMapper composite_volmap
        composite_volume SetProperty composite_prop

proc set_composite_opacity {} {
    	global composite_opacity
        global ren_win3
	tfun RemoveAllPoints
    	tfun AddPoint  20         0.0
    	tfun AddPoint  255        $composite_opacity
	# puts "set_composite_opacity $composite_opacity"
        $ren_win3 Render
}
# -----------------------------
# rendering
# -----------------------------

ren1 AddActor  outline_actor
ren1 AddActor  envelop_actor
ren1 AddActor  plane_cut_actor1
ren1 AddActor  plane_cut_actor2
ren1 AddActor  plane_cut_actor3
#ren1 AddActor  plane_outline_actor1
#ren1 AddActor  plane_outline_actor2
#ren1 AddActor  plane_outline_actor3

ren2 AddActor  outline_actor
ren2 AddVolume mip_volume

ren3 AddActor  outline_actor
ren3 AddVolume composite_volume

ren4 AddActor outline_actor
ren4 AddActor  envelop_actor
ren4 AddActor iso_actor

# ---------------------------------------------------
# axes
# ---------------------------------------------------

vtkCubeAxesActor2D axes_actor
    axes_actor SetInput [reader GetOutput]
    axes_actor SetCamera [ren1 GetActiveCamera]
    axes_actor SetLabelFormat "%6.4g"
if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
    vtkTextProperty axes_text_property
    axes_text_property ShadowOn
    axes_actor SetAxisTitleTextProperty axes_text_property
    axes_actor SetAxisLabelTextProperty axes_text_property
} else {
    axes_actor ShadowOn
}
    axes_actor SetFlyModeToOuterEdges
    axes_actor SetFontFactor 0.8
    eval [axes_actor GetProperty] SetColor $axes_color

ren1 AddProp   axes_actor

# ---------------------------------------------------
# render the image
# ---------------------------------------------------

vtkRenderWindowInteractor iren1;
    iren1 SetRenderWindow $ren_win1;
    #iren1 SetUserMethod {wm deiconify .vtkInteract};
$ren_win1 Render;

vtkRenderWindowInteractor iren2;
    iren2 SetRenderWindow $ren_win2;
    #iren2 SetUserMethod {wm deiconify .vtkInteract};
$ren_win2 Render;

vtkRenderWindowInteractor iren3;
    iren3 SetRenderWindow $ren_win3;
    #iren3 SetUserMethod {wm deiconify .vtkInteract};
$ren_win3 Render;

vtkRenderWindowInteractor iren4;
    iren4 SetRenderWindow $ren_win4;
    #iren4 SetUserMethod {wm deiconify .vtkInteract};
$ren_win4 Render;
# ---------------------------------------------------------------
# prevent the tk window from showing up then start the event loop
# ---------------------------------------------------------------

wm withdraw .

