/*
 *	recite - english text speech synthesizer
 *	Copyright (C) 1993 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 1, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate parameter frames
 */

#include <stdio.h>

#include <error.h>
#include <frame.h>
#include <mem.h>


static	frame_ty	*list;
static	long		list_pos;
static	long		list_max;


void
frame_out(fp)
	frame_ty	*fp;
{
	if (list_pos >= list_max)
	{
		size_t	nbytes;

		list_max += (1L << 10);
		nbytes = list_max * sizeof(frame_ty);
		if (!list)
			list = (frame_ty *)mem_alloc(nbytes);
		else
			mem_change_size((char **)&list, nbytes);
	}
	list[list_pos++] = *fp;
}


void
frame_out_collect(out_p, outlen_p)
	char	**out_p;
	long	*outlen_p;
{
	if (list)
	{
		size_t	nbytes;

		nbytes = list_pos * sizeof(frame_ty);
		if (list_pos < list_max)
			mem_change_size((char **)&list, nbytes);
		*out_p = (char *)list;
		*outlen_p = nbytes;
	}
	else
	{
		*out_p = 0;
		*outlen_p = 0;
	}
	list = 0;
	list_max = 0;
	list_pos = 0;
}


static int calc_width _((int));

static int
calc_width(n)
	int	n;
{
	int	result;

	if (!n)
		return 1;
	if (n < 0)
	{
		result = 1;
		n = -n;
	}
	else
		result = 0;
	while (n > 0)
	{
		n /= 10;
		result++;
	}
	return result;
}


void
frame_write(filename, data, datalen)
	char	*filename;
	char	*data;
	long	datalen;
{       
	int		i;
	FILE		*fp;
	frame_ty	*frame;
	long		frame_count;
	frame_ty	width;

	if (filename)
	{
		fp = fopen(filename, "w");
		if (!fp)
			nfatal("open \"%s\"", filename);
	}
	else
	{
		fp = stdout;
		filename = "(standard output)";
	}
	frame = (frame_ty *)data;
	frame_count = datalen / sizeof(frame_ty);

	/*
	 * compute fields widths
	 */
	width.f0 = 1;
	width.f1 = 1;
	width.f2 = 1;
	width.f3 = 1;
	width.bw1 = 1;
	width.bw2 = 1;
	width.bw3 = 1;
	width.fnz = 1;
	width.fnp = 1;
	width.av = 1;
	width.anp = 1;
	width.asp = 1;
	width.af = 1;
	width.a1 = 1;
	width.a2 = 1;
	width.a3 = 1;
	width.a4 = 1;
	width.a5 = 1;
	width.a6 = 1;
	width.ab = 1;
	for (i = 0; i < frame_count; i++)
	{
		int	w;

#define CWID(xx) w = calc_width(frame[i].xx); if (w > width.xx) width.xx = w;
		CWID(f0);
		CWID(f1);
		CWID(f2);
		CWID(f3);
		CWID(bw1);
		CWID(bw2);
		CWID(bw3);
		CWID(fnz);
		CWID(fnp);
		CWID(av);
		CWID(anp);
		CWID(asp);
		CWID(af);
		CWID(a1);
		CWID(a2);
		CWID(a3);
		CWID(a4);
		CWID(a5);
		CWID(a6);
		CWID(ab);
#undef CWID
	}

	/*
	 * emit the frames
	 */
	for (i = 0; i < frame_count; i++)
	{
		fprintf
		(
			fp,
"%*hd %*hd %*hd %*hd %*hd %*hd %*hd %*hd %*hd %*hd \
%*hd %*hd %*hd %*hd %*hd %*hd %*hd %*hd %*hd %*hd\n",
			width.f0, frame[i].f0,
			width.f1, frame[i].f1,
			width.f2, frame[i].f2,
			width.f3, frame[i].f3,
			width.bw1, frame[i].bw1,
			width.bw2, frame[i].bw2,
			width.bw3, frame[i].bw3,
			width.fnz, frame[i].fnz,
			width.fnp, frame[i].fnp,
			width.av, frame[i].av,
			width.anp, frame[i].anp,
			width.asp, frame[i].asp,
			width.af, frame[i].af,
			width.a1, frame[i].a1,
			width.a2, frame[i].a2,
			width.a3, frame[i].a3,
			width.a4, frame[i].a4,
			width.a5, frame[i].a5,
			width.a6, frame[i].a6,
			width.ab, frame[i].ab
		);
	}
	if (fflush(fp))
		nfatal("write \"%s\"", filename);
	if (fclose(fp))
		nfatal("close \"%s\"", filename);
}


static int frame_read_one _((FILE *fp, char *filename));

static int
frame_read_one(fp, filename)
	FILE		*fp;
	char		*filename;
{
	frame_ty	frame;
	int		n;

	n =
		fscanf
		(
			fp,
		"%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd%hd",
			&frame.f0,
			&frame.f1,
			&frame.f2,
			&frame.f3,
			&frame.bw1,
			&frame.bw2,
			&frame.bw3,
			&frame.fnz,
			&frame.fnp,
			&frame.av,
			&frame.anp,
			&frame.asp,
			&frame.af,
			&frame.a1,
			&frame.a2,
			&frame.a3,
			&frame.a4,
			&frame.a5,
			&frame.a6,
			&frame.ab
		);
	if (n != 20)
	{
		if (ferror(fp))
			nfatal("read \"%s\"", filename);
		if (n != EOF)
			fatal("%s: format error", filename);
		return 0;
	}
	frame_out(&frame);
	return 1;
}


void
frame_read(filename, data, datalen)
	char	*filename;
	char	**data;
	long	*datalen;
{
	FILE	*fp;

	if (filename)
	{
		fp = fopen(filename, "r");
		if (!fp)
			nfatal("open \"%s\"", filename);
	}
	else
	{
		fp = stdin;
		filename = "(standard input)";
	}
	while (frame_read_one(fp, filename))
		;
	if (fclose(fp))
		nfatal("close \"%s\"", filename);
	frame_out_collect(data, datalen);
}
