## SVDLIBC seems to have a bug that leads to incorrect results and infinite loops with
## highly sparse matrices with invariant subspaces, requiring frequent restarts of the Lanczos iteration.
## Here we test these issues with several sample matrices (should be fixed in sparsesvd v0.2)

library(Matrix)
library(sparsesvd)

check.svd <- function (M, tol=1e-2) {
  cat(sprintf("Checking %d x %d matrix %s ...\n", nrow(M), ncol(M), deparse(substitute(M))))
  M.ref <- as.matrix(M)
  res <- sparsesvd(M)
  M.new <- res$u %*% diag(res$d, length(res$d)) %*% t(res$v)
  if (!all(dim(M.new) == dim(M.ref))) stop(sprintf("SVD approximation has wrong dimensions %d x %d", nrow(M.new), ncol(M.new)))
  d <- sqrt(sum((M.new - M.ref)^2))
  cat(sprintf(" --> approximation error ||M.new - M.ref||_2 = %g\n", d))
  if (d > tol) {
    print(round(M.new, 2))
    print(round(M.ref, 2))
    stop("approximation error exceeds tolerance limit")
  }
}

Zero <- as(matrix(0, 3, 5), "dgCMatrix") # all-zero matrix
check.svd(Zero)

D1 <-
new("dgCMatrix", i = 0:1, p = c(0L, 2L), Dim = 2:1, Dimnames = list(
    NULL, NULL), x = c(1, 1), factors = list())
check.svd(D1)

D2 <-
new("dgCMatrix", i = 0:3, p = c(0L, 2L, 4L), Dim = c(4L, 2L), 
    Dimnames = list(NULL, NULL), x = c(1, 1, 1, 1), factors = list())
check.svd(D2)

D5 <-
new("dgCMatrix", i = 0:4, p = 0:5, Dim = c(5L, 5L), Dimnames = list(
    NULL, NULL), x = c(1, 1, 1, 1, 1), factors = list())
check.svd(D5)

D21 <-
new("dgCMatrix", i = 0:21, p = c(0L, 1L, 2L, 3L, 4L, 5L, 6L, 
7L, 8L, 9L, 10L, 12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 
21L, 22L), Dim = 22:21, Dimnames = list(NULL, NULL), x = c(1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
), factors = list())
check.svd(D21)

TermDoc <-
new("dgCMatrix", i = c(0L, 14L, 15L, 13L, 17L, 16L, 4L, 1L, 13L, 
9L, 16L, 6L, 7L, 10L, 9L, 11L, 8L, 14L, 1L, 5L, 16L, 3L, 8L, 
12L, 0L, 14L, 15L, 4L, 3L, 2L, 5L, 2L, 17L, 9L, 3L, 14L, 12L, 
11L, 6L, 7L, 10L, 9L, 3L, 12L), p = c(0L, 3L, 4L, 5L, 6L, 7L, 
9L, 11L, 14L, 16L, 18L, 19L, 21L, 24L, 27L, 28L, 29L, 30L, 31L, 
33L, 34L, 36L, 37L, 38L, 41L, 42L, 44L), Dim = c(18L, 26L), Dimnames = list(
    NULL, NULL), x = c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1), factors = list())
check.svd(TermDoc)

M <-
new("dgTMatrix", i = c(13L, 46L, 377L, 243L, 298L, 345L, 155L, 
422L, 441L, 77L, 167L, 282L, 375L, 389L, 395L, 28L, 30L, 202L, 
250L, 297L, 69L, 84L, 214L, 234L, 317L, 68L, 135L, 196L, 380L, 
414L, 459L, 114L, 229L, 343L, 406L, 70L, 188L, 223L, 322L, 334L, 
429L, 1L, 53L, 96L, 176L, 239L, 296L, 320L, 346L, 371L, 432L, 
27L, 58L, 74L, 198L, 237L, 253L, 256L, 287L, 440L, 97L, 157L, 
185L, 257L, 93L, 116L, 458L, 99L, 110L, 180L, 186L, 333L, 403L, 
219L, 311L, 382L, 49L, 64L, 123L, 171L, 200L, 264L, 335L, 339L, 
357L, 364L, 366L, 448L, 79L, 281L, 417L, 87L, 105L, 128L, 319L, 
398L, 194L, 249L, 276L, 8L, 59L, 72L, 81L, 242L, 267L, 386L, 
454L, 182L, 207L, 226L, 230L, 431L, 24L, 181L, 294L, 305L, 383L, 
410L, 442L, 22L, 103L, 238L, 306L, 336L, 370L, 2L, 23L, 35L, 
119L, 175L, 266L, 350L, 0L, 50L, 149L, 228L, 293L, 397L, 451L, 
17L, 48L, 83L, 215L, 340L, 420L, 456L, 15L, 142L, 277L, 57L, 
150L, 435L, 11L, 62L, 82L, 85L, 164L, 189L, 279L, 246L, 436L, 
75L, 100L, 163L, 165L, 166L, 170L, 247L, 290L, 407L, 433L, 33L, 
131L, 225L, 3L, 19L, 179L, 418L, 4L, 160L, 338L, 355L, 445L, 
452L, 71L, 158L, 183L, 184L, 203L, 331L, 437L, 156L, 199L, 323L, 
94L, 146L, 208L, 424L, 292L, 394L, 425L, 65L, 125L, 205L, 365L, 
396L, 443L, 52L, 136L, 310L, 76L, 204L, 206L, 212L, 299L, 342L, 
356L, 446L, 73L, 95L, 111L, 187L, 341L, 20L, 309L, 325L, 10L, 
31L, 91L, 178L, 252L, 447L, 121L, 197L, 314L, 25L, 169L, 218L, 
221L, 255L, 295L, 378L, 411L, 444L, 44L, 118L, 271L, 173L, 232L, 
374L, 18L, 159L, 210L, 133L, 244L, 369L, 385L, 162L, 312L, 327L, 
363L, 439L, 36L, 313L, 404L, 423L, 153L, 245L, 251L, 303L, 29L, 
248L, 285L, 145L, 261L, 450L, 7L, 45L, 405L, 430L, 14L, 126L, 
286L, 328L, 388L, 434L, 90L, 332L, 354L, 6L, 61L, 152L, 115L, 
134L, 137L, 154L, 174L, 213L, 258L, 324L, 348L, 391L, 415L, 117L, 
138L, 141L, 231L, 259L, 291L, 315L, 438L, 39L, 112L, 321L, 122L, 
260L, 427L, 26L, 40L, 88L, 192L, 268L, 300L, 280L, 283L, 393L, 
413L, 108L, 120L, 151L, 419L, 21L, 211L, 351L, 161L, 209L, 216L, 
161L, 209L, 216L, 98L, 352L, 401L, 104L, 372L, 412L, 358L, 408L, 
409L, 263L, 362L, 449L, 233L, 361L, 402L, 107L, 140L, 222L, 367L, 
9L, 51L, 113L, 132L, 190L, 191L, 195L, 301L, 307L, 330L, 359L, 
376L, 421L, 426L, 86L, 106L, 127L, 42L, 47L, 60L, 262L, 273L, 
304L, 326L, 344L, 32L, 43L, 329L, 272L, 390L, 392L, 240L, 428L, 
460L, 38L, 102L, 109L, 217L, 274L, 349L, 56L, 241L, 453L, 12L, 
37L, 130L, 254L, 80L, 269L, 288L, 384L, 455L, 92L, 172L, 353L, 
101L, 224L, 302L, 318L, 5L, 78L, 89L, 54L, 139L, 177L, 16L, 34L, 
373L, 124L, 284L, 360L, 63L, 227L, 235L, 270L, 308L, 66L, 129L, 
144L, 147L, 337L, 387L, 143L, 220L, 368L, 201L, 316L, 381L, 55L, 
193L, 399L, 67L, 265L, 379L, 400L, 416L, 236L, 275L, 278L, 148L, 
168L, 289L, 41L, 347L, 457L), j = c(0L, 0L, 0L, 1L, 1L, 1L, 2L, 
2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 5L, 5L, 5L, 
5L, 5L, 6L, 6L, 6L, 6L, 6L, 6L, 7L, 7L, 7L, 7L, 8L, 8L, 8L, 8L, 
8L, 8L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 10L, 10L, 10L, 
10L, 10L, 10L, 10L, 10L, 10L, 11L, 11L, 11L, 11L, 12L, 12L, 12L, 
13L, 13L, 13L, 13L, 13L, 13L, 14L, 14L, 14L, 15L, 15L, 15L, 15L, 
15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 16L, 16L, 16L, 17L, 17L, 
17L, 17L, 17L, 18L, 18L, 18L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
19L, 20L, 20L, 20L, 20L, 20L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 
22L, 22L, 22L, 22L, 22L, 22L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
24L, 24L, 24L, 24L, 24L, 24L, 24L, 25L, 25L, 25L, 25L, 25L, 25L, 
25L, 26L, 26L, 26L, 27L, 27L, 27L, 28L, 28L, 28L, 28L, 28L, 28L, 
28L, 29L, 29L, 30L, 30L, 30L, 30L, 30L, 30L, 30L, 30L, 30L, 30L, 
31L, 31L, 31L, 32L, 32L, 32L, 32L, 33L, 33L, 33L, 33L, 33L, 33L, 
34L, 34L, 34L, 34L, 34L, 34L, 34L, 35L, 35L, 35L, 36L, 36L, 36L, 
36L, 37L, 37L, 37L, 38L, 38L, 38L, 39L, 39L, 39L, 40L, 40L, 40L, 
41L, 41L, 41L, 41L, 41L, 41L, 41L, 41L, 42L, 42L, 42L, 42L, 42L, 
43L, 43L, 43L, 44L, 44L, 44L, 44L, 44L, 44L, 45L, 45L, 45L, 46L, 
46L, 46L, 46L, 46L, 46L, 46L, 46L, 46L, 47L, 47L, 47L, 48L, 48L, 
48L, 49L, 49L, 49L, 50L, 50L, 50L, 50L, 51L, 51L, 51L, 51L, 51L, 
52L, 52L, 52L, 52L, 53L, 53L, 53L, 53L, 54L, 54L, 54L, 55L, 55L, 
55L, 56L, 56L, 56L, 56L, 57L, 57L, 57L, 57L, 57L, 57L, 58L, 58L, 
58L, 59L, 59L, 59L, 60L, 60L, 60L, 60L, 60L, 60L, 60L, 60L, 60L, 
60L, 60L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 62L, 62L, 62L, 
63L, 63L, 63L, 64L, 64L, 64L, 64L, 64L, 64L, 65L, 65L, 65L, 65L, 
66L, 66L, 66L, 66L, 67L, 67L, 67L, 68L, 68L, 68L, 69L, 69L, 69L, 
70L, 70L, 70L, 71L, 71L, 71L, 72L, 72L, 72L, 73L, 73L, 73L, 74L, 
74L, 74L, 75L, 75L, 75L, 75L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 
76L, 76L, 76L, 76L, 76L, 76L, 76L, 77L, 77L, 77L, 78L, 78L, 78L, 
78L, 78L, 78L, 78L, 78L, 79L, 79L, 79L, 80L, 80L, 80L, 81L, 81L, 
81L, 82L, 82L, 82L, 82L, 82L, 82L, 83L, 83L, 83L, 84L, 84L, 84L, 
84L, 85L, 85L, 85L, 85L, 85L, 86L, 86L, 86L, 87L, 87L, 87L, 87L, 
88L, 88L, 88L, 89L, 89L, 89L, 90L, 90L, 90L, 91L, 91L, 91L, 92L, 
92L, 92L, 92L, 92L, 93L, 93L, 93L, 93L, 93L, 93L, 94L, 94L, 94L, 
95L, 95L, 95L, 96L, 96L, 96L, 97L, 97L, 97L, 97L, 97L, 98L, 98L, 
98L, 99L, 99L, 99L, 100L, 100L, 100L), Dim = c(461L, 101L), Dimnames = list(
    NULL, NULL), x = c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1), factors = list())
check.svd(M)

A <-
new("dgTMatrix", i = c(0L, 586L, 1958L, 18724L, 10404L, 13340L, 
16234L, 6536L, 21785L, 23539L, 3256L, 7242L, 12660L, 18483L, 
19388L, 19824L, 1238L, 1264L, 8787L, 10708L, 13334L, 2867L, 3512L, 
9360L, 10146L, 14287L, 2854L, 5693L, 8646L, 18887L, 21277L, 27422L, 
4632L, 9888L, 16123L, 20780L, 2960L, 8291L, 9771L, 14585L, 15294L, 
22360L, 26L, 2274L, 3930L, 7690L, 10278L, 13269L, 14444L, 16327L, 
17875L, 22552L, 1207L, 2362L, 3174L, 8653L, 10220L, 10754L, 10857L, 
12861L, 23415L, 3957L, 6679L, 8183L, 10870L, 3773L, 4719L, 26559L, 
4030L, 4486L, 7926L, 8245L, 15243L, 20489L, 9551L, 13960L, 18997L, 
2088L, 2785L, 4965L, 7428L, 8762L, 11435L, 15325L, 15451L, 16967L, 
17323L, 17463L, 25040L, 3368L, 12604L, 21330L, 3615L, 4307L, 
5179L, 14429L, 19969L, 8536L, 10618L, 12325L, 366L, 2394L, 3097L, 
3390L, 10377L, 11606L, 19305L, 25620L, 8051L, 9066L, 9831L, 9923L, 
22547L, 1051L, 8000L, 13148L, 13652L, 19134L, 21014L, 23716L, 
974L, 4140L, 10244L, 13758L, 15363L, 17744L, 110L, 1042L, 1435L, 
4802L, 7663L, 11599L, 16531L, 12L, 2158L, 6172L, 9866L, 13125L, 
19941L, 25208L, 833L, 2072L, 3476L, 9408L, 15581L, 21608L, 25886L, 
749L, 5960L, 12348L, 2342L, 6249L, 23087L, 571L, 2737L, 3420L, 
3576L, 7017L, 8423L, 12401L, 10484L, 23110L, 3212L, 4053L, 7013L, 
7037L, 7194L, 7317L, 10532L, 12917L, 20800L, 22803L, 1374L, 5427L, 
9798L, 178L, 870L, 7869L, 21366L, 181L, 6742L, 15435L, 16834L, 
24213L, 25261L, 3080L, 6701L, 8100L, 8146L, 8845L, 15076L, 23227L, 
6650L, 8667L, 14607L, 3860L, 6059L, 9131L, 21934L, 13039L, 19810L, 
21992L, 2797L, 4977L, 8988L, 17456L, 19865L, 23892L, 2225L, 5696L, 
13948L, 3231L, 8849L, 9013L, 9270L, 13351L, 16093L, 16881L, 24362L, 
3151L, 3911L, 4534L, 8259L, 15656L, 872L, 13889L, 14709L, 512L, 
1303L, 3741L, 7837L, 10750L, 24978L, 4937L, 8651L, 14102L, 1116L, 
7309L, 9485L, 9640L, 10828L, 13251L, 18787L, 21060L, 23901L, 
1879L, 4794L, 11817L, 7440L, 10074L, 18425L, 843L, 6710L, 9218L, 
5651L, 10410L, 17733L, 19290L, 7005L, 13985L, 14964L, 17289L, 
23410L, 1548L, 14100L, 20576L, 21875L, 6356L, 10475L, 10712L, 
13594L, 1261L, 10574L, 12727L, 6047L, 11238L, 25105L, 297L, 1903L, 
20711L, 22391L, 605L, 5002L, 12782L, 14991L, 19368L, 23012L, 
3699L, 15108L, 16802L, 268L, 2562L, 6333L, 4668L, 5682L, 5713L, 
6416L, 7499L, 9318L, 10948L, 14649L, 16450L, 19481L, 21282L, 
4788L, 5758L, 5939L, 9983L, 10954L, 13000L, 14198L, 23381L, 1678L, 
4591L, 14577L, 4958L, 11141L, 22168L, 1128L, 1722L, 3619L, 8484L, 
11661L, 13422L, 12412L, 12688L, 19778L, 21264L, 4379L, 4891L, 
6302L, 21571L, 931L, 9252L, 16600L, 6782L, 9204L, 9435L, 6782L, 
9204L, 9435L, 3959L, 16655L, 20200L, 4153L, 18020L, 21108L, 17079L, 
20885L, 20922L, 11384L, 17273L, 25071L, 10100L, 17218L, 20241L, 
4331L, 5925L, 9736L, 17639L, 446L, 2200L, 4603L, 5442L, 8431L, 
8464L, 8566L, 13425L, 13761L, 15026L, 17106L, 18624L, 21649L, 
22001L, 3603L, 4316L, 5073L, 1775L, 1984L, 2414L, 11242L, 12045L, 
13619L, 14769L, 16229L, 1337L, 1844L, 14997L, 11948L, 19464L, 
19777L, 10280L, 22259L, 27458L, 1666L, 4138L, 4406L, 9461L, 12200L, 
16461L, 2339L, 10325L, 25264L, 580L, 1646L, 5412L, 10793L, 3385L, 
11791L, 12875L, 19187L, 25846L, 3749L, 7436L, 16780L, 4107L, 
9793L, 13445L, 14305L, 255L, 3257L, 3674L, 2279L, 5912L, 7777L, 
761L, 1404L, 18184L, 4970L, 12711L, 17107L, 2760L, 9858L, 10187L, 
11794L, 13807L, 2813L, 5403L, 5986L, 6112L, 15382L, 19313L, 5979L, 
9579L, 17654L, 8784L, 14210L, 18958L, 2315L, 8532L, 19996L, 2826L, 
11510L, 18848L, 20043L, 21313L, 10212L, 12304L, 12367L, 6160L, 
7248L, 12897L, 1764L, 16440L, 26230L), j = c(0L, 0L, 0L, 0L, 
1L, 1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 
4L, 5L, 5L, 5L, 5L, 5L, 6L, 6L, 6L, 6L, 6L, 6L, 7L, 7L, 7L, 7L, 
8L, 8L, 8L, 8L, 8L, 8L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 11L, 11L, 11L, 11L, 
12L, 12L, 12L, 13L, 13L, 13L, 13L, 13L, 13L, 14L, 14L, 14L, 15L, 
15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 16L, 16L, 
16L, 17L, 17L, 17L, 17L, 17L, 18L, 18L, 18L, 19L, 19L, 19L, 19L, 
19L, 19L, 19L, 19L, 20L, 20L, 20L, 20L, 20L, 21L, 21L, 21L, 21L, 
21L, 21L, 21L, 22L, 22L, 22L, 22L, 22L, 22L, 23L, 23L, 23L, 23L, 
23L, 23L, 23L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 25L, 25L, 25L, 
25L, 25L, 25L, 25L, 26L, 26L, 26L, 27L, 27L, 27L, 28L, 28L, 28L, 
28L, 28L, 28L, 28L, 29L, 29L, 30L, 30L, 30L, 30L, 30L, 30L, 30L, 
30L, 30L, 30L, 31L, 31L, 31L, 32L, 32L, 32L, 32L, 33L, 33L, 33L, 
33L, 33L, 33L, 34L, 34L, 34L, 34L, 34L, 34L, 34L, 35L, 35L, 35L, 
36L, 36L, 36L, 36L, 37L, 37L, 37L, 38L, 38L, 38L, 39L, 39L, 39L, 
40L, 40L, 40L, 41L, 41L, 41L, 41L, 41L, 41L, 41L, 41L, 42L, 42L, 
42L, 42L, 42L, 43L, 43L, 43L, 44L, 44L, 44L, 44L, 44L, 44L, 45L, 
45L, 45L, 46L, 46L, 46L, 46L, 46L, 46L, 46L, 46L, 46L, 47L, 47L, 
47L, 48L, 48L, 48L, 49L, 49L, 49L, 50L, 50L, 50L, 50L, 51L, 51L, 
51L, 51L, 51L, 52L, 52L, 52L, 52L, 53L, 53L, 53L, 53L, 54L, 54L, 
54L, 55L, 55L, 55L, 56L, 56L, 56L, 56L, 57L, 57L, 57L, 57L, 57L, 
57L, 58L, 58L, 58L, 59L, 59L, 59L, 60L, 60L, 60L, 60L, 60L, 60L, 
60L, 60L, 60L, 60L, 60L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 
62L, 62L, 62L, 63L, 63L, 63L, 64L, 64L, 64L, 64L, 64L, 64L, 65L, 
65L, 65L, 65L, 66L, 66L, 66L, 66L, 67L, 67L, 67L, 68L, 68L, 68L, 
69L, 69L, 69L, 70L, 70L, 70L, 71L, 71L, 71L, 72L, 72L, 72L, 73L, 
73L, 73L, 74L, 74L, 74L, 75L, 75L, 75L, 75L, 76L, 76L, 76L, 76L, 
76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 77L, 77L, 77L, 
78L, 78L, 78L, 78L, 78L, 78L, 78L, 78L, 79L, 79L, 79L, 80L, 80L, 
80L, 81L, 81L, 81L, 82L, 82L, 82L, 82L, 82L, 82L, 83L, 83L, 83L, 
84L, 84L, 84L, 84L, 85L, 85L, 85L, 85L, 85L, 86L, 86L, 86L, 87L, 
87L, 87L, 87L, 88L, 88L, 88L, 89L, 89L, 89L, 90L, 90L, 90L, 91L, 
91L, 91L, 92L, 92L, 92L, 92L, 92L, 93L, 93L, 93L, 93L, 93L, 93L, 
94L, 94L, 94L, 95L, 95L, 95L, 96L, 96L, 96L, 97L, 97L, 97L, 97L, 
97L, 98L, 98L, 98L, 99L, 99L, 99L, 100L, 100L, 100L), Dim = c(95474L, 
101L), Dimnames = list(NULL, NULL), x = c(0, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1), factors = list())
check.svd(A)
