//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Job/JobViewActivities.cpp
//! @brief     Implements class JobViewActivities
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Job/JobViewActivities.h"
#include "Base/Util/Assert.h"
#include "GUI/View/Tool/mainwindow_constants.h"

namespace {

QMap<JobViewActivity, QVector<JobViewFlags::Dock>> createActivityMap()
{
    QMap<JobViewActivity, QVector<JobViewFlags::Dock>> result;
    result[JobViewActivity::JobView] = QVector<JobViewFlags::Dock>() << JobViewFlags::JOB_LIST_DOCK;
    result[JobViewActivity::RealTime] = QVector<JobViewFlags::Dock>()
                                        << JobViewFlags::JOB_LIST_DOCK
                                        << JobViewFlags::REAL_TIME_DOCK;
    result[JobViewActivity::Fitting] =
        QVector<JobViewFlags::Dock>()
        << JobViewFlags::JOB_LIST_DOCK << JobViewFlags::REAL_TIME_DOCK
        << JobViewFlags::FIT_PANEL_DOCK << JobViewFlags::JOB_MESSAGE_DOCK;
    return result;
}

QMap<JobViewActivity, QVector<JobViewFlags::Dock>> activityToDocks = createActivityMap();

QMap<JobViewActivity, QString> activityNames = {
    {JobViewActivity::JobView, GUI::Constants::JobViewActivityName},
    {JobViewActivity::RealTime, GUI::Constants::JobRealTimeActivityName},
    {JobViewActivity::Fitting, GUI::Constants::JobFittingActivityName}};

} // namespace

namespace JobViewActivities {

//! Returns list of all available activities.

QList<JobViewActivity> all()
{
    return activityNames.keys();
}

//! Returns name of activity.

QString nameFromActivity(JobViewActivity activity)
{
    QMap<JobViewActivity, QString>::const_iterator it = activityNames.find(activity);
    ASSERT(it != activityNames.end());
    return it.value();
}

//! Returns activity by its name.

JobViewActivity activityFromName(QString name)
{
    ASSERT(activityNames.values().contains(name));
    return activityNames.key(name);
}

//! Returns vector of JobView's dockId which have to be shown for given activity.

QVector<JobViewFlags::Dock> activeDocks(JobViewActivity activity)
{
    QMap<JobViewActivity, QVector<JobViewFlags::Dock>>::const_iterator it =
        activityToDocks.find(activity);
    ASSERT(it != activityToDocks.end());
    return it.value();
}

} // namespace JobViewActivities
