//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Beam/BeamDistributionItem.cpp
//! @brief     Implements class BeamDistributionItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Beam/BeamDistributionItem.h"
#include "GUI/Model/Descriptor/DistributionItems.h"
#include "Param/Distrib/Distributions.h"

namespace {
namespace Tag {

const QString Distribution("Distribution");
const QString ExpandGroupbox("ExpandGroupbox");

} // namespace Tag
} // namespace

BeamDistributionItem::~BeamDistributionItem() = default;

void BeamDistributionItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // distribution
    w->writeStartElement(Tag::Distribution);
    m_distribution.writeTo(w);
    w->writeEndElement();

    // groupbox: is expanded?
    w->writeStartElement(Tag::ExpandGroupbox);
    XML::writeAttribute(w, XML::Attrib::value, m_expandGroupBox);
    w->writeEndElement();
}

void BeamDistributionItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // distribution
        if (tag == Tag::Distribution) {
            m_distribution.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // groupbox: is expanded?
        } else if (tag == Tag::ExpandGroupbox) {
            XML::readAttribute(r, XML::Attrib::value, &m_expandGroupBox);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

double BeamDistributionItem::meanValue() const
{
    std::unique_ptr<IDistribution1D> domainDistr = createDistribution1D();
    if (domainDistr)
        return domainDistr->mean() / scaleFactor();
    return dynamic_cast<SymmetricResolutionItem*>(distributionItem())->mean();
}

void BeamDistributionItem::resetToValue(double value)
{
    auto* d = new DistributionNoneItem();
    d->setMean(value);
    m_distribution.setCurrentItem(d);
}

double BeamDistributionItem::scaleFactor() const
{
    return 1.0;
}

std::unique_ptr<IDistribution1D> BeamDistributionItem::createDistribution1D() const
{
    return distributionItem()->createDistribution(scaleFactor());
}
