/*
blahtex: a TeX to MathML converter designed with MediaWiki in mind
blahtexml: an extension of blahtex with XML processing in mind
http://gva.noekeon.org/blahtexml

Copyright (c) 2006, David Harvey
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the names of the authors nor the names of their affiliation may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef BLAHTEX_MANAGER_H
#define BLAHTEX_MANAGER_H

#include <string>
#include <vector>
#include <memory>
#include <set>
#include "Misc.h"
#include "MathmlNode.h"
#include "LayoutTree.h"
#include "ParseTree.h"

namespace blahtex
{

// The Manager class coordinates all the bits and pieces required to convert
// the given TeX input into MathML and purified TeX output, including
// tokenising, texvc-compatiblity macros, building the parse and layout
// trees, deciding which LaTeX packages to include, converting mathvariant
// to MathML version 1 fonts.
//
// The Manager class could be used as an interface between the blahtex core
// and an external program; alternatively, the Interface class (see
// Interface.h) provides a simpler interface.

class Manager
{
public:
    Manager();

    // ProcessInput generates a parse tree and a layout tree from the
    // supplied input.
    //
    // If texvcCompatibility is set, then ProcessInput will append a series
    // of macros to emulate various non-standard commands that texvc
    // recognises (see gTexvcCompatibilityMacros). This corresponds to the
    // command line option "--texvc-compatible-commands".
    void ProcessInput(
        const std::wstring& input,
        bool texvcCompatibility = false,
        bool displayStyle = false
    );

    // GenerateMathml generates a XML tree containing MathML markup.
    // Returns the root node.
    std::auto_ptr<MathmlNode> GenerateMathml(
        const MathmlOptions& options
    ) const;

    // GeneratePurifiedTex returns a string containing a complete TeX file
    // (including any required \usepackage commands) that could be fed to
    // LaTeX to produce a graphical version of the input.
    std::wstring GeneratePurifiedTex(
        const PurifiedTexOptions& options
    ) const;

    // GeneratePurifiedTexOnly returns a string containing only
    // the equation in LaTeX
    std::wstring GeneratePurifiedTexOnly() const;

    // A few accessor functions.
    const ParseTree::MathNode* GetParseTree() const
    {
        return mParseTree.get();
    }

    const LayoutTree::Node* GetLayoutTree() const
    {
        return mLayoutTree.get();
    }

private:
    // These store the parse tree and layout tree generated by ProcessInput.
    std::auto_ptr<ParseTree::MathNode> mParseTree;
    std::auto_ptr<LayoutTree::Node> mLayoutTree;

    // This flag is set if the user has requested "strict spacing" rules
    // (see SpacingControl) via the magic "\strictspacing" command.
    bool mStrictSpacingRequested;
    
    // There are a handful of errors that get picked up during the layout
    // tree building phase, but which we want to return as MathML-related
    // errors; i.e. we can still run PNG generation. If one of these
    // happens, we cache it in mDelayedMathmlError, and return it when
    // someone tries to GenerateMathml().
    // FIX: this is a bit hacky and badly designed.
    // Come back and fix it up one day.
    bool mHasDelayedMathmlError;
    Exception mDelayedMathmlError;

    // gStandardMacros is a string which, in effect, gets inserted at the
    // beginning of any input string handled by ProcessInput. It contains
    // a sequence of macro definitions ("\newcommand"s) which set up some
    // standard TeX synonyms.
    static std::wstring gStandardMacros;

    // gTexvcCompatibilityMacros is similar; it contains definitions for
    // commands recognised by texvc but that are not standard TeX/LaTeX/
    // AMS-LaTeX. (See also the texvcCompatibility flag.)
    static std::wstring gTexvcCompatibilityMacros;

    // Tokenised version of gStandardMacros and gTexvcCompatibilityMacros
    // (computed only once, when first used):
    static std::vector<std::wstring> gStandardMacrosTokenised;
    static std::vector<std::wstring> gTexvcCompatibilityMacrosTokenised;
};

}

#endif

// end of file @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
