/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::common::f_fmla;
use crate::polyeval::f_polyeval5;
use crate::sin_cosf::sincosf_eval::sincosf_eval;

/// Computes sin(x) - x
pub fn f_sinmxf(x: f32) -> f32 {
    let x_abs = x.to_bits() & 0x7fff_ffffu32;
    let xd = x as f64;
    let ax: u32 = x.to_bits().wrapping_shl(1);
    if ax == 0 || ax >= 0xffu32 << 24 {
        if x_abs == 0u32 {
            // For signed zeros.
            return x;
        }
        return f32::NAN; // x == inf or x == NaN
    }

    // |x| <= pi/16
    if x_abs <= 0x3e49_0fdbu32 {
        // |x| < 0.000443633
        if x_abs < 0x39e8_9769u32 {
            // sin(x) - x ~ -x^3/6 + x^5/120 + O(x^7)
            let dx = x as f64;
            let x2 = dx * dx;
            let c = f_fmla(
                x2,
                f64::from_bits(0x3f81111111111111),
                f64::from_bits(0xbfc5555555555555),
            ) * x2;
            return (c * dx) as f32;
        }

        let xsqr = xd * xd;

        // Generated by Sollya:
        // d = [2^-10, pi/16];
        // f_sinmx = (sin(x) - x)/x;
        // Q = fpminimax(f_sinmx, [|0, 2, 4, 6, 8|], [|D...|], d);
        // See ./notes/sinmxf.sollya
        let p = f_polyeval5(
            xsqr,
            f64::from_bits(0xbb8f6a339382f07f),
            f64::from_bits(0xbfc5555555555545),
            f64::from_bits(0x3f811111110ddf5c),
            f64::from_bits(0xbf2a019fb330c620),
            f64::from_bits(0x3ec719bc2a614884),
        );
        return (xd * p) as f32;
    }

    // Formula:
    //   sin(x) = sin((k + y)*pi/32)
    //          = sin(y*pi/32) * cos(k*pi/32) + cos(y*pi/32) * sin(k*pi/32)
    // The values of sin(k*pi/32) and cos(k*pi/32) for k = 0..31 are precomputed
    // and stored using a vector of 32 doubles. Sin(y*pi/32) and cos(y*pi/32) are
    // computed using degree-7 and degree-6 minimax polynomials generated by
    // Sollya respectively.

    let rs = sincosf_eval(xd, x_abs);
    let sinx = f_fmla(rs.sin_y, rs.cos_k, f_fmla(rs.cosm1_y, rs.sin_k, rs.sin_k));
    (sinx - xd) as f32
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn f_sinf_test() {
        assert_eq!(f_sinmxf(0.00016344387), -0.00000000000072770376);
        assert_eq!(f_sinmxf(0.0), 0.0);
        assert_eq!(f_sinmxf(-0.0), 0.0);
        assert_eq!(f_sinmxf(1.0), -0.15852901);
        assert_eq!(f_sinmxf(0.3), -0.004479794);
        assert_eq!(f_sinmxf(-1.0), 0.15852901);
        assert_eq!(f_sinmxf(-0.3), 0.004479794);
        assert_eq!(f_sinmxf(std::f32::consts::PI / 2.), -0.5707964);
        assert!(f_sinmxf(f32::INFINITY).is_nan());
        assert!(f_sinmxf(f32::NEG_INFINITY).is_nan());
        assert!(f_sinmxf(f32::NAN).is_nan());
    }
}
