#!/bin/bash
#
# SPDX-License-Identifier: GPL-2.0-only
#
# Copyright (C) 2019 Philippe Proulx <pproulx@efficios.com>
#

# This test validates that a `src.ctf.fs` component successfully reads
# specific CTF traces and creates the expected messages.
#
# Such CTF traces to open either exist (in `tests/ctf-traces/1/succeed`)
# or are generated by this test using local trace generators.

SH_TAP=1

if [ -n "${BT_TESTS_SRCDIR:-}" ]; then
	UTILSSH="$BT_TESTS_SRCDIR/utils/utils.sh"
else
	UTILSSH="$(dirname "$0")/../../../utils/utils.sh"
fi

# shellcheck source=../../../utils/utils.sh
source "$UTILSSH"

this_dir_relative="plugins/src.ctf.fs/succeed"
this_dir_build="$BT_TESTS_BUILDDIR/$this_dir_relative"
data_dir="$BT_TESTS_DATADIR/$this_dir_relative"

test_ctf_common_details_args=("-p" "with-trace-name=no,with-stream-name=no")

# Print the expected stdout file for test with name `$1`, CTF version
# `$2` and MIP version `$3`.
find_expect_file() {
	local test_name="$1"
	local ctf_version="$2"
	local mip_version="$3"

	names=(
		"$data_dir/trace-$test_name-ctf$ctf_version-mip$mip_version.expect"
		"$data_dir/trace-$test_name-ctf$ctf_version.expect"
		"$data_dir/trace-$test_name-mip$mip_version.expect"
		"$data_dir/trace-$test_name.expect"
	)

	for name in "${names[@]}"; do
		if [[ -f "$name" ]]; then
			echo "$name"
			return
		fi
	done

	echo "Could not find expect file for test $test_name, CTF $ctf_version, MIP $mip_version" >&2
	exit 1
}

test_ctf_gen_single() {
	name="$1"

	diag "Generating trace '$name'"
	bt_diff_details_ctf_gen_single "$this_dir_build/gen-trace-$name" \
		"$data_dir/trace-$name.expect" \
		"${test_ctf_common_details_args[@]}" "-p" "with-uuid=no,with-uid=no"
	ok $? "Generated trace '$name' gives the expected output"
}

# Parameters: <trace-name> <ctf-version>
test_ctf_single_version() {
	local name="$1"
	local ctf_version="$2"
	local trace_path="$BT_CTF_TRACES_PATH/$ctf_version/succeed/$name"
	local expected_stdout

	for mip_version in 0 1; do
		if ! bt_is_valid_ctf_mip_combo "$ctf_version" $mip_version; then
			continue;
		fi

		local expected_stdout

		expected_stdout=$(find_expect_file "$name" "$ctf_version" $mip_version)

		diag "CTF $ctf_version, MIP $mip_version, expect file $expected_stdout"
		bt_diff_details_ctf_single \
			"$expected_stdout" \
			"$trace_path" \
			--allowed-mip-versions=$mip_version \
			"${test_ctf_common_details_args[@]}"
		ok $? "CTF $ctf_version: MIP $mip_version: Trace '$name' gives the expected output"
	done
}

test_ctf_single() {
	local name="$1"

	test_ctf_single_version "$name" 1
	test_ctf_single_version "$name" 2
}

test_packet_end() {
	local name="$1"

	for ctf_version in 1 2; do
		local trace_path="$BT_CTF_TRACES_PATH/$ctf_version/succeed/$name"

		for mip_version in 0 1; do
			if ! bt_is_valid_ctf_mip_combo $ctf_version $mip_version; then
				continue;
			fi

			local expected_stdout
			local details_comp=("-c" "sink.text.details")
			local details_args=("-p" "with-trace-name=no,with-stream-name=no,with-metadata=no,compact=yes")
			local temp_stdout_output_file
			local temp_greped_stdout_output_file
			local temp_stderr_output_file

			expected_stdout=$(find_expect_file "$name" $ctf_version $mip_version)
			temp_stdout_output_file="$(mktemp -t actual-stdout.XXXXXX)"
			temp_greped_stdout_output_file="$(mktemp -t greped-stdout.XXXXXX)"
			temp_stderr_output_file="$(mktemp -t actual-stderr.XXXXXX)"

			bt_cli "$temp_stdout_output_file" "$temp_stderr_output_file" \
			"$trace_path" "${details_comp[@]}" "${details_args[@]}"

			bt_grep "Packet end" "$temp_stdout_output_file" > "$temp_greped_stdout_output_file"

			diag "CTF version $ctf_version, MIP $mip_version, expected file $expected_stdout"
			bt_diff "$expected_stdout" "$temp_greped_stdout_output_file"
			ok $? "CTF $ctf_version: Trace '$name' gives the expected stdout"

			bt_diff /dev/null "$temp_stderr_output_file"
			ok $? "CTF $ctf_version: Trace '$name' gives the expected stderr"

			rm -f "$temp_stdout_output_file" "$temp_stderr_output_file" "$temp_greped_stdout_output_file"
		done
	done
}

test_force_origin_unix_epoch() {
	local name1="$1"
	local name2="$2"
	local src_ctf_fs_args=("-p" "force-clock-class-origin-unix-epoch=true")
	local details_comp=("-c" "sink.text.details")
	local details_args=("-p" "with-trace-name=no,with-stream-name=no,with-metadata=yes,compact=yes")
	local temp_stdout_output_file
	local temp_stderr_output_file

	temp_stdout_output_file="$(mktemp -t actual-stdout.XXXXXX)"
	temp_stderr_output_file="$(mktemp -t actual-stderr.XXXXXX)"

	for ctf_version in 1 2; do
		local trace_1_path="$BT_CTF_TRACES_PATH/$ctf_version/succeed/$name1"
		local trace_2_path="$BT_CTF_TRACES_PATH/$ctf_version/succeed/$name2"

		for mip_version in 0 1; do
			if ! bt_is_valid_ctf_mip_combo $ctf_version $mip_version; then
				continue;
			fi

			local expected_stdout

			expected_stdout=$(find_expect_file "$name1-$name2" $ctf_version $mip_version)

			diag "CTF $ctf_version, MIP $mip_version, expected file $expected_stdout"
			bt_cli "$temp_stdout_output_file" "$temp_stderr_output_file" \
				--allowed-mip-versions=$mip_version \
				"$trace_1_path" "${src_ctf_fs_args[@]}" \
				"$trace_2_path" "${src_ctf_fs_args[@]}" \
				"${details_comp[@]}" "${details_args[@]}"

			bt_diff "$expected_stdout" "$temp_stdout_output_file"
			ok $? "CTF $ctf_version: MIP $mip_version: Trace '$name1' and '$name2' give the expected stdout"

			bt_diff /dev/null "$temp_stderr_output_file"
			ok $? "CTF $ctf_version: MIP $mip_version: Trace '$name1' and '$name2' give the expected stderr"

			rm -f "$temp_stdout_output_file" "$temp_stderr_output_file"
		done
	done
}

test_clock_offset_goes_back_in_time() {
	local trace_name="clock-offset-goes-back-in-time"
	local tmp_dir

	tmp_dir=$(mktemp -d -t "test-$trace_name.XXXXXXX")

	bt_gen_mctf_trace "$data_dir/$trace_name/chunk1.mctf" "$tmp_dir/chunk1"
	bt_gen_mctf_trace "$data_dir/$trace_name/chunk2.mctf" "$tmp_dir/chunk2"

	bt_diff_details_ctf_single \
		"$data_dir/trace-$trace_name.expect" \
		"$tmp_dir" \
		-p "with-stream-name=no"
	ok $? "Trace '$trace_name' gives the expected output"

	rm -rf "$tmp_dir"
}

plan_tests 47

test_force_origin_unix_epoch 2packets barectf-event-before-packet
test_ctf_gen_single simple
test_ctf_single smalltrace
test_ctf_single 2packets
test_ctf_single barectf-event-before-packet
test_ctf_single session-rotation
test_ctf_single lttng-tracefile-rotation
test_ctf_single array-align-elem
test_ctf_single struct-array-align-elem
test_ctf_single meta-ctx-sequence
test_ctf_single_version meta-clk-cls-before-trace-cls 2
test_ctf_single_version def-clk-freq 1

test_packet_end lttng-event-after-packet
test_packet_end lttng-crash

test_clock_offset_goes_back_in_time
